import { ref, watch, onMounted, Ref } from 'vue'

let darkModeRef: Ref<boolean> | null = null

export const useDarkMode = () => {
  // Create dark mode ref only once
  if (!darkModeRef) {
    darkModeRef = ref(false)

    // Initialize dark mode from localStorage on mount
    if (typeof window !== 'undefined') {
      const savedTheme = localStorage.getItem('theme')
      if (savedTheme === 'dark' || (!savedTheme && window.matchMedia('(prefers-color-scheme: dark)').matches)) {
        darkModeRef.value = true
        document.documentElement.classList.add('dark')
        document.body.classList.add('dark')
      } else {
        darkModeRef.value = false
        document.documentElement.classList.remove('dark')
        document.body.classList.remove('dark')
      }
    }

    // Watch for changes and apply to document
    watch(darkModeRef, (newValue) => {
      if (newValue) {
        document.documentElement.classList.add('dark')
        document.body.classList.add('dark')
        localStorage.setItem('theme', 'dark')
      } else {
        document.documentElement.classList.remove('dark')
        document.body.classList.remove('dark')
        localStorage.setItem('theme', 'light')
      }
    })
  }

  const toggleDarkMode = () => {
    if (darkModeRef) {
      darkModeRef.value = !darkModeRef.value
    }
  }

  return {
    isDarkMode: darkModeRef,
    toggleDarkMode
  }
}
